import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Headphones, Calendar, MessageSquare, Plus, Minus } from 'lucide-react'
import { useNavigate } from 'react-router-dom'
import ImageSlider from '../components/ImageSlider'
import './MyRentals.css'

const MyRentals = () => {
  const navigate = useNavigate()
  const [rentals, setRentals] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('musicRentals')
    if (saved) {
      try {
        setRentals(JSON.parse(saved))
      } catch (e) {
        setRentals([])
      }
    } else {
      const defaultRentals = [
        {
          id: 1,
          equipment: 'Fender Stratocaster',
          date: '2025-01-25',
          review: 'Отличная гитара, звук превосходный!',
          photo: '/images/rental-1.jpg'
        },
        {
          id: 2,
          equipment: 'Yamaha P-125',
          date: '2025-01-20',
          review: 'Клавиши очень отзывчивые, рекомендую',
          photo: '/images/rental-2.jpg'
        },
        {
          id: 3,
          equipment: 'Shure SM58',
          date: '2025-01-15',
          review: 'Качественный микрофон, без нареканий',
          photo: '/images/rental-3.jpg'
        }
      ]
      setRentals(defaultRentals)
      localStorage.setItem('musicRentals', JSON.stringify(defaultRentals))
    }
  }, [])

  return (
    <div className="my-rentals-minimal">
      <div className="container">
        <motion.section 
          className="hero-minimal-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.8 }}
        >
          <div className="hero-minimal-content">
            <motion.div 
              className="hero-number-minimal"
              initial={{ opacity: 0, x: -30 }}
              animate={{ opacity: 1, x: 0 }}
              transition={{ delay: 0.2 }}
            >
              01
            </motion.div>
            <div className="hero-text-minimal">
              <motion.h1 
                className="hero-title-minimal"
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.3 }}
              >
                Мои аренды
              </motion.h1>
              <motion.p 
                className="hero-description-minimal"
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ delay: 0.4 }}
              >
                Управляйте арендованным оборудованием и отслеживайте сроки
              </motion.p>
            </div>
            <motion.button 
              className="cta-minimal-button"
              onClick={() => navigate('/rent')}
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ delay: 0.5 }}
              whileHover={{ scale: 1.02 }}
              whileTap={{ scale: 0.98 }}
            >
              <Plus size={16} strokeWidth={1.5} />
              <span>Новая аренда</span>
            </motion.button>
          </div>
        </motion.section>

        <ImageSlider />

        <section className="rentals-list-minimal">
          {rentals.length === 0 ? (
            <motion.div 
              className="empty-state-minimal"
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
            >
              <div className="empty-line-minimal"></div>
              <h2 className="empty-title-minimal">Нет активных аренд</h2>
              <p className="empty-text-minimal">Начните с аренды первого инструмента</p>
              <button className="empty-button-minimal" onClick={() => navigate('/rent')}>
                Арендовать
              </button>
            </motion.div>
          ) : (
            <div className="rentals-timeline-minimal">
              {rentals.map((rental, index) => {
                const date = new Date(rental.date).toLocaleDateString('ru-RU', {
                  day: 'numeric',
                  month: 'long',
                  year: 'numeric'
                })

                return (
                  <motion.div
                    key={rental.id}
                    className="rental-item-minimal"
                    initial={{ opacity: 0, x: -30 }}
                    animate={{ opacity: 1, x: 0 }}
                    transition={{ delay: 0.2 + index * 0.1 }}
                  >
                    <div className="rental-line-minimal"></div>
                    <div className="rental-content-minimal">
                      <div className="rental-image-minimal">
                        <img src={rental.photo} alt={rental.equipment} />
                      </div>
                      <div className="rental-info-minimal">
                        <div className="rental-header-minimal">
                          <h3 className="rental-title-minimal">{rental.equipment}</h3>
                          <div className="rental-date-minimal">
                            <Calendar size={14} strokeWidth={1.5} />
                            <span>{date}</span>
                          </div>
                        </div>
                        {rental.review && (
                          <div className="rental-review-minimal">
                            <MessageSquare size={14} strokeWidth={1.5} />
                            <span>{rental.review}</span>
                          </div>
                        )}
                      </div>
                    </div>
                  </motion.div>
                )
              })}
            </div>
          )}
        </section>

        <section className="stats-minimal-section">
          <motion.div 
            className="stats-grid-minimal"
            initial={{ opacity: 0 }}
            whileInView={{ opacity: 1 }}
            viewport={{ once: true }}
            transition={{ duration: 0.6 }}
          >
            <div className="stat-item-minimal">
              <div className="stat-value-minimal">{rentals.length}</div>
              <div className="stat-label-minimal">Аренд</div>
            </div>
            <div className="stat-divider-minimal"></div>
            <div className="stat-item-minimal">
              <div className="stat-value-minimal">24/7</div>
              <div className="stat-label-minimal">Поддержка</div>
            </div>
            <div className="stat-divider-minimal"></div>
            <div className="stat-item-minimal">
              <div className="stat-value-minimal">100%</div>
              <div className="stat-label-minimal">Качество</div>
            </div>
          </motion.div>
        </section>
      </div>
    </div>
  )
}

export default MyRentals
